﻿/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;

using System.Data;
using System.Configuration;
using System.Web.SessionState;

//our data access library
using DataAccess;


//MDWS web service
using MDWSLib.MDWSEmrSvc;

/// <summary>
/// CMDWSTransfe is used to transfer data from MDWS to the central db
/// </summary>
public class CMDWSTransfer: CData
{
    const long MATCH_TYPE_LAST_NAME = 1;
    const long MATCH_TYPE_LSSN = 2;

    //Constructor
    public CMDWSTransfer(CData data)
        : base(data)
    {
        //constructors are not inherited in c#!
    }

    /// <summary>
    /// US:5729 US:5349
    /// Transfers 1 patient and adds the patient to a team, ward etc... as needed
    /// </summary>
    /// <param name="tpa"></param>
    /// <param name="lCount"></param>
    /// <returns></returns>
    public CStatus TransferPatient(HttpSessionState WebSession, 
                                    string strKey, 
                                    long lMatchType, 
                                    string strMatch, 
                                    PatientTO toPatient)
    {
        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //nothing to do
        if (toPatient == null)
        {
            return status;
        }

        //check for valid patient
        if (toPatient == null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            status.StatusComment = "Patient is null";

            return status;
        }

        //make sure patient is valid (no fault)
        if (toPatient.fault != null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            status.StatusComment = "Patient has a fault";

            return status;
        }

        if (String.IsNullOrEmpty(toPatient.ssn) ||
            toPatient.ssn.IndexOf("*") != -1)
        {
            PatientTO toPatSel = GetMDWSSOAPClient().select(toPatient.localPid);

            //if the user does not have access to select the patient a fault will occur
            //in this case we dont want to clobber the original selected patient
            if (toPatSel.fault == null)
            {
                toPatient = toPatSel;
            }
        }

        //our patient item
        CPatientDataItemXfer pdi = new CPatientDataItemXfer();

        //get the home address
        AddressTO homeAddress = toPatient.homeAddress;
        string strHomeCity = String.Empty;
        string strHomeState = String.Empty;
        string strHomeAddr1 = String.Empty;
        string strHomeAddr2 = String.Empty;
        string strHomeZipCode = String.Empty;
        if (homeAddress != null)
        {
            if (homeAddress.fault == null)
            {
                strHomeCity = homeAddress.city;
                strHomeState = homeAddress.state;
                strHomeAddr1 = homeAddress.streetAddress1;
                strHomeAddr2 = homeAddress.streetAddress2;
                strHomeState = homeAddress.state;
                strHomeZipCode = homeAddress.zipcode;
            }
        }
        pdi.HomeAddr1 = strHomeAddr1;
        pdi.HomeAddr2 = strHomeAddr2;
        pdi.HomeState = strHomeState;
        pdi.HomeCity = strHomeCity;
        pdi.HomeZipCode = strHomeZipCode;

        //get the home phone
        //our mask is (212)432-1555
        string strHomeAreaCode = String.Empty;
        string strHomeNumber = String.Empty;
        string strFullHomeNumber = String.Empty;
        PhoneNumTO homePhone = toPatient.homePhone;
        if (homePhone != null)
        {
            if (homePhone.fault == null)
            {
                strHomeAreaCode = homePhone.areaCode;
                strHomeNumber = homePhone.number;
                strFullHomeNumber = "(" + strHomeAreaCode + ")" + strHomeNumber;
            }
        }

        pdi.HomePhone = strFullHomeNumber;

        string strEthnicity = toPatient.ethnicity;


        //formatted 19450309
        pdi.DOB = CDataUtils2.GetMDWSShortDate(toPatient.dob);

        //name
        pdi.FullName = toPatient.patientName;
        string[] splitFullName = toPatient.patientName.Split(',');
        if (splitFullName[0] != null)
        {
            pdi.LastName = splitFullName[0];
        }

        if (splitFullName.Length > 1)
        {
            if (splitFullName[1] != null)
            {
                pdi.FirstName = splitFullName[1];
            }
        }

        pdi.MI = String.Empty;

        //gender
        //default gender to unknown
        pdi.Sex = k_SEX.UNKNOWN;
        if (toPatient.gender == null)
        {
            pdi.Sex = k_SEX.UNKNOWN;
        }
        else
        {
            if (toPatient.gender.ToLower().IndexOf("unk") != -1)
            {
                pdi.Sex = k_SEX.UNKNOWN;
            }
            else
            {
                if (toPatient.gender.ToUpper() == "M")
                {
                    pdi.Sex = k_SEX.MALE;
                }
                if (toPatient.gender.ToUpper() == "F")
                {
                    pdi.Sex = k_SEX.FEMALE;
                }
            }
        }

        pdi.SSN = toPatient.ssn;

        //DFN (also called PID, IEN, Local PID): This is the IEN in the PATIENT file for the 
        //patient at a single Vista site. Remember, the VA has many Vista sites and vets may 
        //visit multiple facilities. The IEN is a unique value for a record in a Vista file at a 
        //site. So, the DFN is the patient's unique identifier at a site. Furthermore, the 
        //composite key: sitecode + DFN is a unique value across the enterprise. For example, 
        //DFN 12345 at site 506 (Ann Arbor): 50612345 is a unique identifier for this context. 
        //You can find a patient's DFN through the MATCH function. You pass the DFN to the 
        //SELECT function to "select" your patient when using MDWS.
        pdi.PatientID = toPatient.localPid;
        
        //MPI PID: 
        //Until the magic of the MPI (now called MVI...), it was tough to link patients 
        //across the VA. If a snowbird needed a checkup in Florida, how would the doc know 
        //for sure the records from Buffalo for SMITH,JOHN were the same person? 
        //The MPI assigns a VA unique identifier to every patient at registration. 
        //Every site a patient has visited contains this value. Also, a little know fact, 
        //every site also contains the DFNs for every other site a patient has visited. 
        //You only need to know one site a patient has visited to know them all. 
        //This is part of the magic that enables the wonderful call "setupMultiSiteQuery". 
        //Call that after you select a patient and MDWS will visit every other site the patient 
        //has visited, and select the patient. All subsequent calls will, in parallel, be executed 
        //at each of those connected sites. I WAS TOLD NOT TO USE MPI PID


        //match is different in revamp, exact last name match
        if (lMatchType == MATCH_TYPE_LAST_NAME)
        {
            if (pdi.LastName.ToUpper() != strMatch.ToUpper())
            {
                //we do not need to transfer this patient so just return success!
                return new CStatus();
            }
        }

        //LSSN
        if (lMatchType == MATCH_TYPE_LSSN)
        {
            if (strMatch.Length != 5)
            {
                return new CStatus();
            }

            if (pdi.SSN.Length != 9)
            {
                return new CStatus();
            }

            if (pdi.LastName.Length < 2)
            {
                return new CStatus();
            }

            if (strMatch.Substring(1, 4) != pdi.SSN.Substring(5, 4))
            {
                return new CStatus();
            }

            if (strMatch.Substring(0, 1) != pdi.LastName.Substring(0, 1))
            {
                return new CStatus();
            }
        }

        //save the patient to our db
        string strPatID = string.Empty;
        CPatientDataXfer dta = new CPatientDataXfer(this);
        //added patient id and encounter id incase we need to insert a new patient
        status = dta.SavePatient((long)k_SYSTEM.MDWS,
                                 strKey,
                                 WebSession,
                                 pdi,
                                 out strPatID);

        return status;
    }


    /// <summary>
    /// US:5716
    /// Transfers 1 patient and returns the central db patient id
    /// </summary>
    /// <param name="tpa"></param>
    /// <param name="lCount"></param>
    /// <returns></returns>
    public CStatus TransferPatient(HttpSessionState WebSession,
                                    string strKey,
                                    PatientTO toPatient,
                                    out string strPatID)
    {
        strPatID = "";

        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //nothing to do
        if (toPatient == null)
        {
            return status;
        }

        //make sure patient is valid (no fault)
        if (toPatient.fault != null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            status.StatusComment = "Patient has a fault";

            return status;
        }

        if (String.IsNullOrEmpty(toPatient.ssn) ||
            toPatient.ssn.IndexOf("*") != -1)
        {
            PatientTO toPatSel = GetMDWSSOAPClient().select(toPatient.localPid);

            //if the user does not have access to select the patient a fault will occur
            //in this case we dont want to clobber the original selected patient
            if (toPatSel.fault == null)
            {
                toPatient = toPatSel;
            }
        }

        //our patient item
        CPatientDataItemXfer pdi = new CPatientDataItemXfer();

        //get the home address
        AddressTO homeAddress = toPatient.homeAddress;
        string strHomeCity = String.Empty;
        string strHomeState = String.Empty;
        string strHomeAddr1 = String.Empty;
        string strHomeAddr2 = String.Empty;
        string strHomeZipCode = String.Empty;
        if (homeAddress != null)
        {
            if (homeAddress.fault == null)
            {
                strHomeCity = homeAddress.city;
                strHomeState = homeAddress.state;
                strHomeAddr1 = homeAddress.streetAddress1;
                strHomeAddr2 = homeAddress.streetAddress2;
                strHomeState = homeAddress.state;
                strHomeZipCode = homeAddress.zipcode;
            }
        }
        pdi.HomeAddr1 = strHomeAddr1;
        pdi.HomeAddr2 = strHomeAddr2;
        pdi.HomeState = strHomeState;
        pdi.HomeCity = strHomeCity;
        pdi.HomeZipCode = strHomeZipCode;

        //get the home phone
        //our mask is (212)432-1555
        string strHomeAreaCode = String.Empty;
        string strHomeNumber = String.Empty;
        string strFullHomeNumber = String.Empty;
        PhoneNumTO homePhone = toPatient.homePhone;
        if (homePhone != null)
        {
            if (homePhone.fault == null)
            {
                strHomeAreaCode = homePhone.areaCode;
                strHomeNumber = homePhone.number;
                strFullHomeNumber = "(" + strHomeAreaCode + ")" + strHomeNumber;
            }
        }

        pdi.HomePhone = strFullHomeNumber;

        string strEthnicity = toPatient.ethnicity;


        //formatted 19450309
        pdi.DOB = CDataUtils2.GetMDWSShortDate(toPatient.dob);

        //name
        pdi.FullName = toPatient.patientName;
        string[] splitFullName = toPatient.patientName.Split(',');
        if (splitFullName[0] != null)
        {
            pdi.LastName = splitFullName[0];
        }

        if (splitFullName.Length > 1)
        {
            if (splitFullName[1] != null)
            {
                pdi.FirstName = splitFullName[1];
            }
        }

        pdi.MI = String.Empty;

        //gender
        //default gender to unknown
        pdi.Sex = k_SEX.UNKNOWN;
        if (toPatient.gender == null)
        {
            pdi.Sex = k_SEX.UNKNOWN;
        }
        else
        {
            if (toPatient.gender.ToLower().IndexOf("unk") != -1)
            {
                pdi.Sex = k_SEX.UNKNOWN;
            }
            else
            {
                if (toPatient.gender.ToUpper() == "M")
                {
                    pdi.Sex = k_SEX.MALE;
                }
                if (toPatient.gender.ToUpper() == "F")
                {
                    pdi.Sex = k_SEX.FEMALE;
                }
            }
        }

        pdi.SSN = toPatient.ssn;

        //DFN (also called PID, IEN, Local PID): This is the IEN in the PATIENT file for the 
        //patient at a single Vista site. Remember, the VA has many Vista sites and vets may 
        //visit multiple facilities. The IEN is a unique value for a record in a Vista file at a 
        //site. So, the DFN is the patient's unique identifier at a site. Furthermore, the 
        //composite key: sitecode + DFN is a unique value across the enterprise. For example, 
        //DFN 12345 at site 506 (Ann Arbor): 50612345 is a unique identifier for this context. 
        //You can find a patient's DFN through the MATCH function. You pass the DFN to the 
        //SELECT function to "select" your patient when using MDWS.
        pdi.PatientID = toPatient.localPid;

        //MPI PID: 
        //Until the magic of the MPI (now called MVI...), it was tough to link patients 
        //across the VA. If a snowbird needed a checkup in Florida, how would the doc know 
        //for sure the records from Buffalo for SMITH,JOHN were the same person? 
        //The MPI assigns a VA unique identifier to every patient at registration. 
        //Every site a patient has visited contains this value. Also, a little know fact, 
        //every site also contains the DFNs for every other site a patient has visited. 
        //You only need to know one site a patient has visited to know them all. 
        //This is part of the magic that enables the wonderful call "setupMultiSiteQuery". 
        //Call that after you select a patient and MDWS will visit every other site the patient 
        //has visited, and select the patient. All subsequent calls will, in parallel, be executed 
        //at each of those connected sites.

        //save the patient to our db
        CPatientDataXfer dta = new CPatientDataXfer(this);
        //added patient id and encounter id incase we need to insert a new patient
        status = dta.SavePatient((long)k_SYSTEM.MDWS,
                                 strKey,
                                 WebSession,
                                 pdi,
                                 out strPatID);

        return status;
    }

    /// <summary>
    /// US:5729 US:5349
    /// Transfers a patient array to our db
    /// </summary>
    /// <param name="tpa"></param>
    /// <param name="lCount"></param>
    /// <returns></returns>
    public CStatus TransferPatientArray(HttpSessionState WebSession,
                                        TaggedPatientArray tpa,
                                        string strKey,
                                        long lMatchType, //1=lastname, 2=LSSN
                                        string strMatch,
                                        out long lCount)
    {
        //initialize parameters
        lCount = 0;

        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //nothing to do
        if (tpa == null || tpa.patients == null)
        {
            return status;
        }

        //check to see if there is anything to transfer
        if (tpa.count < 1)
        {
            return status;
        }

        foreach (PatientTO pat in tpa.patients)
        {
            //patient transfer object to work with
            PatientTO patTO = pat;

            //additional demographics
            DemographicSetTO[] demog = pat.demographics;

            //In some cases the patients are coming back with missing data
            //needed for the transfer. If you select the patient the data fills in
            //and transfer will work
            if (pat.ssn == null || 
                String.IsNullOrEmpty(pat.ssn) ||
                pat.ssn.IndexOf("*") != -1)
            {
                PatientTO toPatSel = GetMDWSSOAPClient().select(pat.localPid);

                //if the user does not have access to select the patient a fault will occur
                //in this case we dont want to clobber the original selected patient
                if (toPatSel.fault == null)
                {
                    patTO = toPatSel;
                }
            }

            //example: patientName: fortynine,Inpatient
            //
            //we go by lastname only so screen out records we definately do not need to transfer
            // to our db, also we only transfer patients that have a valid name and ssn
            //
            if (patTO.patientName != null)
            {
                if (patTO.patientName != string.Empty)
                {
                    if (patTO.ssn != null)
                    {
                        if (patTO.ssn != string.Empty)
                        {
                            string strNameCheck = pat.patientName.ToUpper();
                            string strSrch = strMatch.ToUpper() + ",";
                            int nPos = strNameCheck.IndexOf(strSrch);

                            //only try to transfer patients matching last name 
                            //or we are doing an LSSN
                            if ((nPos >= 0) || (lMatchType == 2))
                            {
                                CStatus statXfer = new CStatus();
                                statXfer = TransferPatient(WebSession, strKey, lMatchType, strMatch, patTO);
                                lCount++;
                                if (!statXfer.Status)
                                {
                                    //todo: better error message
                                    status.Status = false;
                                    status.StatusCode = k_STATUS_CODE.Failed;
                                    status.StatusComment = "One or more of your patient(s) could not be transfered!";
                                    //note: dont return yet, try the rest of the patients
                                }
                            }
                        }
                    }
                }
            }
        }

        return status;
    }



    EmrSvcSoapClient m_EmrSvcSoapClient = null;
    /// <summary>
    /// gets the soap client so we can talk to mdws
    /// </summary>
    /// <returns></returns>
    EmrSvcSoapClient GetMDWSSOAPClient()
    {
        m_EmrSvcSoapClient = new EmrSvcSoapClient("EmrSvcSoap");
        return m_EmrSvcSoapClient;

        //this is used from communicator also and we 
        //have no "Session" in communicator
     /*   if (WebSession != null)
        {
            if (WebSession["EmrSvcSoapClient"] == null)
            {
                m_EmrSvcSoapClient = new EmrSvcSoapClient("EmrSvcSoap");
                WebSession["EmrSvcSoapClient"] = m_EmrSvcSoapClient;
            }

            return (EmrSvcSoapClient)WebSession["EmrSvcSoapClient"];
        }
        else
        {
            if (m_EmrSvcSoapClient == null)
            {
                m_EmrSvcSoapClient = new EmrSvcSoapClient("EmrSvcSoap");
            }

            return m_EmrSvcSoapClient;
        }*/
    }

    /// <summary>
    /// US:838
    /// get mdws patients
    /// </summary>
    /// <param name="lPatCLID"></param>
    /// <param name="ds"></param>
    /// <returns></returns>
    public CStatus GetMDWSPatientDS(out DataSet ds)
    {
        //initialize parameters
        ds = null;

        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //load the paramaters list
        CDataParameterList pList = new CDataParameterList(base.SessionID,
                                                  base.ClientIP,
                                                  base.UserID);
        //get the dataset
        CDataSet cds = new CDataSet();
        status = cds.GetOracleDataSet(base.DBConn,
                                      "PCK_MDWS.GetMDWSPatientRS",
                                      pList,
                                      out ds);
        if (!status.Status)
        {
            return status;
        }

        return status;
    }

    /// <summary>
    /// US:5348 transfer regions and sites from MDWS to our tables
    /// </summary>
    /// <param name="ra"></param>
    /// <returns></returns>
    public CStatus TransferSites(RegionArray ra)
    {
        CStatus status = new CStatus();

        //nothing to do
        if (ra == null || ra.regions == null)
        {
            return status;
        }

        foreach (RegionTO regionbTo in ra.regions)
        {
            string strRegionID = regionbTo.id;
            if (String.IsNullOrEmpty(strRegionID))
            {
                strRegionID = "";
            }

            string strRegionName = regionbTo.name;
            if (String.IsNullOrEmpty(strRegionName))
            {
                strRegionName = "";
            }
            
            //save the region
            CSiteDataXfer siteData = new CSiteDataXfer(this);
            siteData.SaveRegion((long)k_SYSTEM.MDWS,
                               CDataUtils2.ToLong(strRegionID),
                                strRegionName);
            
            for (int i = 0; i < regionbTo.sites.count; i++)
            {
                string strSiteCode = regionbTo.sites.sites[i].sitecode;
                if (String.IsNullOrEmpty(strSiteCode))
                {
                    strSiteCode = "";
                }

                string strSiteName = regionbTo.sites.sites[i].displayName;
                if (String.IsNullOrEmpty(strSiteName))
                {
                    strSiteName = "";
                }

                //save the site
                siteData.SaveSite((long)k_SYSTEM.MDWS,
                                 CDataUtils2.ToLong(strRegionID),
                                 CDataUtils2.ToLong(strSiteCode),
                                  strSiteName);

            }
        }


        return status;

    }


    /// <summary>
    /// US:1945 transfers patient notes from MDWS to our tables
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="strNoteTitle"></param>
    /// <param name="nLookBackDays"></param>
    /// <returns></returns>
    /*public CStatus TransferPatientNotes(string strPatientID,
                                        string strNoteTitle,
                                        long lItemID,
                                        int nLookBackDays)
    {
        CStatus status = new CStatus();

        //select the patient
        PatientTO toSelect = GetMDWSSOAPClient().select(strPatientID);
        
        //nothing to do
        if (toSelect == null)
        {
            return status;
        }

        //end if we have a fault
        if (toSelect.fault != null)
        {
            return new CMDWSStatus(toSelect.fault);
        }
        
        try
        {
            //todate is today
            DateTime dtToDate = DateTime.Now;
            string strToDate =CDataUtils2.GetMilitaryDateAsString(dtToDate);

            //fromdate is today - lookback
            DateTime dtFromDate = dtToDate.AddDays(-1 * nLookBackDays);
            string strFromDate =CDataUtils2.GetMilitaryDateAsString(dtFromDate);

            //todo: testing, our new notes are not signed so it appears that they cant be 
            //retrieved from getNotesWithText but they do retrieve by using getUnsignedNotes
            //TaggedNoteArrays tnaUnsigned = GetMDWSSOAPClient().getUnsignedNotes(strFromDate,
            //                                                                    strToDate,
            //                                                                    0);

            //get the notes
            TaggedNoteArrays tnas = GetMDWSSOAPClient().getNotesWithText(strFromDate,
                                                                         strToDate,
                                                                         0);
            if (tnas == null)
            {
                return status;
            }
            
            if (tnas.fault != null)
            {
                return new CMDWSStatus(toSelect.fault);
            }

            foreach (TaggedNoteArray tna in tnas.arrays)
            {
                if (tna.notes != null)
                {
                    foreach (NoteTO note in tna.notes)
                    {
                        string strLocalT = note.localTitle.ToLower().Trim();
                        strLocalT = strLocalT.Replace("<", "[");
                        strLocalT = strLocalT.Replace(">", "]");

                        if (strLocalT ==
                            strNoteTitle.ToLower().Trim())
                        {
                            //create a new patient item
                            CPatientItemDataItem di = new CPatientItemDataItem();
                            di.PatientID = strPatientID;
                            di.ItemID = lItemID;
                            di.SourceTypeID = (long)k_SOURCE_TYPE_ID.VistA;
                            di.EntryDate =CDataUtils2.GetMDWSDate(note.timestamp);
                            
                            //build a pat item component to hold the note text
                            CPatientItemComponentDataItem cdi = new CPatientItemComponentDataItem();
                            cdi.PatientID = strPatientID;
                            cdi.ItemID = lItemID;

                            //need to get the component given the item
                            DataSet dsItemComps = null;
                            CItemComponentData cid = new CItemComponentData(this);
                            cid.GetItemComponentDS(lItemID, k_ACTIVE_ID.All, out dsItemComps);
                            if (dsItemComps != null)
                            {
                                //note titles only have 1 component.
                                cdi.ComponentID =CDataUtils2.GetDSLongValue(dsItemComps, "ITEM_COMPONENT_ID");
                            }


                            string strNoteText = note.text;

                            //todo: remove this temp fix when we switch to clob
                            if (strNoteText.Length > 4000)
                            {
                                strNoteText = strNoteText.Substring(0, 3995);
                                strNoteText += "...";
                            }

                            cdi.ComponentValue = strNoteText;
                            cdi.EntryDate = di.EntryDate;
                                                        
                            //create a new item component list with 1 component 
                            //for the note text
                            CPatientItemCompList comps = new CPatientItemCompList();
                            comps.Add(cdi);

                            //check to see if the note already exists
                            //in the vappct database, if so we want to update 
                            //it, else insert it
                            bool bInsert = true;
                            DataSet dsExists = null;
                            CPatientItemData itemData = new CPatientItemData(this);
                            status = itemData.GetPatItemCompDS( strPatientID,
                                                                di.ItemID,
                                                                di.EntryDate,
                                                                out dsExists);
                            if (!status.Status)
                            {
                                //todo:
                                return status;
                            }

                            if (!CDataUtils2.IsEmpty(dsExists))
                            {
                                bInsert = false;
                            }

                            if (!bInsert)
                            {
                                di.PatItemID =CDataUtils2.GetDSLongValue(dsExists, "pat_item_id");
                                cdi.PatItemID = di.PatItemID;
                                
                                //todo: just testing
                                //cdi.ComponentValue = "UPDATED: " + cdi.ComponentValue;

                                //update the record 
                                status = itemData.UpdatePatientItem(di,
                                                                    comps);
                                if (!status.Status)
                                {
                                    //todo:
                                }

                    //            CPatientChecklistLogic pcll = new CPatientChecklistLogic(this);
                      //          pcll.RunLogic(di.PatientID, di.ItemID);
                            }
                            else
                            {
                                //add the record (insert only todo need to
                                //handle dupes in the sp maybe have it update)
                                long lPatItemID = 0;
                                status = itemData.InsertPatientItem(di,
                                                                    comps,
                                                                    out lPatItemID);
                                if (!status.Status)
                                {
                                    //todo:
                                }

                        //        CPatientChecklistLogic pcll = new CPatientChecklistLogic(this);
                          //      pcll.RunLogic(di.PatientID, di.ItemID);
                            }

                            //cleanup
                            di = null;
                            cdi = null;
                            comps = null;
                            itemData = null;
                        }
                    }
                }
            }
        }
        catch (Exception e)
        {
            status.StatusCode = k_STATUS_CODE.Failed;
            status.StatusComment = e.Message;
            return status;
        }


        //now that we inserted all the items run the logic
        CPatientChecklistLogic pcll = new CPatientChecklistLogic(this);
        pcll.RunLogic(strPatientID, lItemID);

        return status;
    }
     */

    /// <summary>
    /// US:852 transfer patient labs from MDWS
    /// </summary>
    /// <param name="strPatientID"></param>
    /// <param name="strNoteTitle"></param>
    /// <param name="lItemID"></param>
    /// <param name="nLookBackDays"></param>
    /// <returns></returns>
   /*
    * public CStatus TransferPatientLabs( string strPatientID,
                                        string strMapID,
                                        long lItemID,
                                        int nLookBackDays)
    {
        CStatus status = new CStatus();

        //select the patient
        PatientTO toSelect = GetMDWSSOAPClient().select(strPatientID);

        //nothing to do
        if (toSelect == null)
        {
            return status;
        }

        //end if we have a fault
        if (toSelect.fault != null)
        {
            return new CMDWSStatus(toSelect.fault);
        }

        try
        {
            //todate is today
            DateTime dtToDate = DateTime.Now;
            string strToDate =CDataUtils2.GetMilitaryDateAsString(dtToDate);

            //fromdate is today - lookback
            DateTime dtFromDate = dtToDate.AddDays(-1 * nLookBackDays);
            string strFromDate =CDataUtils2.GetMilitaryDateAsString(dtFromDate);

            //map id may have a carrot dilimiter so need to strip it
            string strMap = strMapID;
            string[] strMapSplit = strMap.Split('^');
            if(strMapSplit.Length > 1)
            {
                strMap = strMapSplit[0].ToString();
            }
            
            //get the lab results for this patient
            TaggedChemHemRptArrays ChemHemArrays =
                GetMDWSSOAPClient().getChemHemReports( strFromDate,
                                                       strToDate,
                                                       0);

            if (ChemHemArrays.fault == null)
            {
                for (int i = 0; i < ChemHemArrays.count; i++)
                {
                    TaggedChemHemRptArray ChemHemArray = ChemHemArrays.arrays[i];
                    if (ChemHemArray.rpts != null)
                    {
                        foreach (ChemHemRpt rpt in ChemHemArray.rpts)
                        {
                            foreach (LabResultTO result in rpt.results)
                            {
                                LabTestTO toLab = result.test;
                                if (toLab.fault == null)
                                {
                                    //if id of lab = map then process it
                                    if (toLab.id == strMap)
                                    {
                                        //create a new patient item
                                        CPatientItemDataItem di = new CPatientItemDataItem();
                                        di.PatientID = strPatientID;
                                        di.ItemID = lItemID;
                                        di.SourceTypeID = (long)k_SOURCE_TYPE_ID.VistA;
                                        
                                        //use the report time stamp
                                        di.EntryDate =CDataUtils2.GetMDWSDate(rpt.timestamp);

                                        //build a pat item component to hold the note text
                                        CPatientItemComponentDataItem cdi = new CPatientItemComponentDataItem();
                                        cdi.PatientID = strPatientID;
                                        cdi.ItemID = lItemID;

                                        //need to get the component given the item
                                        DataSet dsItemComps = null;
                                        CItemComponentData cid = new CItemComponentData(this);
                                        cid.GetItemComponentDS(lItemID, k_ACTIVE_ID.All, out dsItemComps);
                                        if (dsItemComps != null)
                                        {
                                            //note titles only have 1 component.
                                            cdi.ComponentID =CDataUtils2.GetDSLongValue(dsItemComps, "ITEM_COMPONENT_ID");
                                        }

                                        //get the result
                                        string strResult = result.value;
                                        //todo: remove this temp fix when we switch to clob
                                        if (strResult.Length > 4000)
                                        {
                                            strResult = strResult.Substring(0, 3995);
                                            strResult += "...";
                                        }
                                        
                                        //component value = the result
                                        cdi.ComponentValue = strResult;

                                        //component date is same as item date
                                        cdi.EntryDate = di.EntryDate;
                                        
                                        //create a new item component list with 1 component 
                                        //for the note text
                                        CPatientItemCompList comps = new CPatientItemCompList();
                                        comps.Add(cdi);

                                        //check to see if the note already exists
                                        //in the vappct database, if so we want to update 
                                        //it, else insert it
                                        bool bInsert = true;
                                        DataSet dsExists = null;
                                        CPatientItemData itemData = new CPatientItemData(this);
                                        status = itemData.GetPatItemCompDS(strPatientID,
                                                                            di.ItemID,
                                                                            di.EntryDate,
                                                                            out dsExists);
                                        if (!status.Status)
                                        {
                                            //todo:
                                            return status;
                                        }

                                        if (!CDataUtils2.IsEmpty(dsExists))
                                        {
                                            bInsert = false;
                                        }

                                        if (!bInsert)
                                        {
                                            di.PatItemID =CDataUtils2.GetDSLongValue(dsExists, "pat_item_id");
                                            cdi.PatItemID = di.PatItemID;

                                            //update the record 
                                            status = itemData.UpdatePatientItem(di,
                                                                                comps);
                                            if (!status.Status)
                                            {
                                                //todo:
                                            }

                                  //          CPatientChecklistLogic pcll = new CPatientChecklistLogic(this);
                                    //        pcll.RunLogic(di.PatientID, di.ItemID);
                                        }
                                        else
                                        {
                                            //add the record (insert only todo need to
                                            //handle dupes in the sp maybe have it update)
                                            long lPatItemID = 0;
                                            status = itemData.InsertPatientItem(di,
                                                                                comps,
                                                                                out lPatItemID);
                                            if (!status.Status)
                                            {
                                                //todo:
                                            }

                                  //          CPatientChecklistLogic pcll = new CPatientChecklistLogic(this);
                                    //        pcll.RunLogic(di.PatientID, di.ItemID);
                                        }

                                        //cleanup
                                        di = null;
                                        cdi = null;
                                        comps = null;
                                        itemData = null;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        catch (Exception e)
        {
            status.StatusCode = k_STATUS_CODE.Failed;
            status.StatusComment = e.Message;
            return status;
        }


        //now that we inserted all the items run the logic
        CPatientChecklistLogic pcll = new CPatientChecklistLogic(this);
        pcll.RunLogic(strPatientID, lItemID);

        return status;
    }      */


    /// <summary>
    /// transfer lab tests from MDWS to the VAPPCT database
    /// </summary>
    /// <param name="la"></param>
    /// <param name="lCount"></param>
    /// <returns></returns>
    public CStatus TransferLabTestArray(TaggedLabTestArrays la, 
                                        out long lCount,
                                        out string strNextSearch)
    {
        //create a status object and check for valid dbconnection
        lCount = 0;
        strNextSearch = String.Empty;

        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //check for valid teams
        if (la == null)
        {
           status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            status.StatusComment = "lab array is null";

            return status;
        }

        //make sure patient is valid (no fault)
        if (la.fault != null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            string strErr = "Error transferring lab tests, if this error continues please contact your system administrator.";
            status.StatusComment = strErr;

            //per wasa scan do not return sensitive information to the user.
            //status.StatusComment = la.fault.message;
            return status;
        }

        //loop over the teams and save each one to the database
        string strNames = String.Empty;
        foreach (TaggedLabTestArray ta in la.arrays)
        {
            if (ta != null)
            {
                if (ta.labTests != null)
                {
                    if (ta.labTests.tests != null)
                    {
                        foreach (LabTestTO toLab in ta.labTests.tests)
                        {
                            if (toLab != null)
                            {
                                lCount++;

                                string strLOINC = String.Empty;
                                if (toLab.loinc != null)
                                {
                                    strLOINC = toLab.loinc;
                                }

                                string strLabTestID = String.Empty; 
                                if (toLab.id != null)
                                {
                                    strLabTestID = toLab.id;
                                }

                                string strName = String.Empty;
                                if (toLab.name != null)
                                {
                                    strName = toLab.name;
                                }

                                strNextSearch = strName;

                                string strHIREF = String.Empty;
                                if (toLab.hiRef != null)
                                {
                                    strHIREF = toLab.hiRef;
                                }

                                string strLOREF = String.Empty;
                                if (toLab.lowRef != null)
                                {
                                    strLOREF = toLab.lowRef;
                                }

                                string strREFRANGE = String.Empty;
                                if (toLab.refRange != null)
                                {
                                    strREFRANGE = toLab.refRange;
                                }

                                string strUnits = String.Empty;
                                if (toLab.units != null)
                                {
                                    strUnits = toLab.units;
                                }
                                
                                string strDescription = String.Empty;

                                //get the lab test description using a seperate call
                                TaggedTextArray tta = GetMDWSSOAPClient().getLabTestDescription(toLab.id);
                                if (tta != null)
                                {
                                    if (tta.fault == null)
                                    {
                                        if (tta.results[0] != null)
                                        {
                                            strDescription = tta.results[0].text;
                                        }
                                    }
                                }
                        
                                //save the lab test to th vappct database
                                CLabDataXfer ld = new CLabDataXfer(this);
                                status = ld.SaveLabTest((long)k_SYSTEM.MDWS, 
                                                         strLabTestID,
                                                         strName,
                                                         strHIREF,
                                                         strLOREF,
                                                         strREFRANGE,
                                                         strUnits,
                                                         strDescription,
                                                         strLOINC);
                                if (!status.Status)
                                {
                                    //todo: dont end because 1 does not save...
                                }
                            }
                        }
                    }
                }
            }
        }
        return status;
    }
    
    /// <summary>
    /// US:838
    /// Transfer MDWS teams to our db
    /// </summary>
    /// <param name="txtTeams"></param>
    /// <returns></returns>
    public CStatus TransferTeams(TaggedText txtTeams, out long lCount)
    {

        //create a status object and check for valid dbconnection
        lCount = 0;
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //check for valid teams
        if (txtTeams == null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            status.StatusComment = "teams is null";

            return status;
        }

        //make sure patient is valid (no fault)
        if (txtTeams.fault != null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            string strErr = "Error transferring teams, if this error continues please contact your system administrator.";
            status.StatusComment = strErr;

            //per wasa scan do not return sensitive information to the user.
            //status.StatusComment = txtTeams.fault.message;
            return status;
        }

        //loop over the teams and save each one to the database
        foreach (TaggedText txtTeam in txtTeams.taggedResults)
        {
            if (txtTeam != null)
            {
                lCount++;

                long lTeamID =CDataUtils2.ToLong(txtTeam.tag);
                CTeamDataXfer td = new CTeamDataXfer(this);
                status = td.SaveTeam((long)k_SYSTEM.MDWS, lTeamID, txtTeam.text);

                if (!status.Status)
                {
                    
                }
            }
        }

        return status;
    }
    
    /// <summary>
    /// US:6002
    /// Transfer MDWS note tiles to our db
    /// </summary>
    /// <param name="txtTeams"></param>
    /// <returns></returns>
    public CStatus TransferNoteTitles(TaggedTextArray tta,
                                      string strTarget)
    {
        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //nothing to do
        if (tta == null || tta.results == null)
        {
            return status;
        }

        //make string upper for compare below
        string strSearch = strTarget.ToUpper();

        //when the matches come back they start with the note titles that match the search then
        //the list continues on with note titles that do not match. this means we need to check for
        //an actual match and quit transferring when we do not find one!

        //bool to signal we are done searching
        bool bDone = true;

        //loop over alll taggedreulsts in the array
        foreach (TaggedText tt in tta.results)
        {
            //loop and process this block of note titles
            foreach (TaggedText ttr in tt.taggedResults)
            {
                //get the note title
                long lNoteTitleTag = CDataUtils2.ToLong(ttr.tag);

                foreach (string strNoteLabel in ttr.textArray)
                {
                    string strCheck = strNoteLabel.ToUpper();
                    int nPos = strCheck.IndexOf(strSearch);
                    if (nPos != -1)
                    {
                        //check to see if the note tile is a consult
                        long lIsConsult = 0;
               
                        TextTO ttoConsult = GetMDWSSOAPClient().isConsultNote(ttr.tag);
                        if (ttoConsult != null)
                        {
                            if (ttoConsult.text != null)
                            {
                                if (ttoConsult.text.Trim().ToUpper() == "Y")
                                {
                                    lIsConsult = 1;
                                }
                            }
                        }

                        //do not transfer isOneVisit notes
                        //need visit info wont work from here
                        //GetMDWSSOAPClient().isOneVisitNote(ttr.tag, strNoteLabel, 

                        //flag as isperf note
                        long lIsPerf = 0;
                        TextTO ttoPerf = GetMDWSSOAPClient().isPrfNote(ttr.tag);
                        if (ttoPerf != null)
                        {
                            if (ttoPerf.text != null)
                            {
                                if (ttoPerf.text.Trim().ToUpper() == "Y")
                                {
                                    lIsPerf = 1;
                                }
                            }
                        }


                        //< > cause page validation problems.
                        string strNoteLbl = strNoteLabel;
                        strNoteLbl = strNoteLbl.Replace("<", "[");
                        strNoteLbl = strNoteLbl.Replace(">", "]");

                        //save the note title in our db
                        CNoteTitleDataXfer ntd = new CNoteTitleDataXfer(this);
                        status = ntd.SaveNoteTitle((long)k_SYSTEM.MDWS,
                                                   lNoteTitleTag,
                                                   strNoteLbl,
                                                   lIsConsult);

                        bDone = false;
                    }
                   
                }
            }
        }

        if (bDone)
        {
            //match not found so we are done
            status.StatusCode = k_STATUS_CODE.Done;
            status.StatusComment = "";
        }
        
        return status;
    }

    /// <summary>
    /// US:838
    /// transfer MDWS specialties
    /// </summary>
    /// <param name="txtSpecialties"></param>
    /// <param name="lCount"></param>
    /// <returns></returns>
    public CStatus TransferSpecialties(TaggedText txtSpecialties,
                                       out long lCount)
    {
        //create a status object and check for valid dbconnection
        lCount = 0;
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //check for valid teams
        if (txtSpecialties == null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            status.StatusComment = "specialties is null";

            return status;
        }

        //make sure patient is valid (no fault)
        if (txtSpecialties.fault != null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            string strErr = "Error transferring specialties, if this error continues please contact your system administrator.";
            status.StatusComment = strErr;

            //per wasa scan do not return sensitive information to the user.
            //status.StatusComment = txtSpecialties.fault.message;
            return status;
        }

        //loop over the teams and save each one to the database
        foreach (TaggedText txtSpecialty in txtSpecialties.taggedResults)
        {
            if (txtSpecialty != null)
            {
                lCount++;

                long lSpecialtyID =CDataUtils2.ToLong(txtSpecialty.tag);
                CSpecialtyDataXfer sd = new CSpecialtyDataXfer(this);
                status = sd.SaveSpecialty((long)k_SYSTEM.MDWS, lSpecialtyID, txtSpecialty.text);

                if (!status.Status)
                {
                    
                }
            }
        }

        return status;
    }

    /// <summary>
    /// US:6005 transfer patient appointments
    /// </summary>
    /// <param name="taas"></param>
    /// <returns></returns>
    public CStatus TransferPatientAppointments( TaggedAppointmentArrays taas)
    {
        CStatus status = new CStatus();

        //nothing to do
        if (taas == null || taas.arrays == null)
        {
            return status;
        }

        foreach (TaggedAppointmentArray taa in taas.arrays)
        {
            for (int i = 0; i < taa.count; i++)
            {
                AppointmentTO ato = taa.appts[i];
                if (ato == null || taas.fault != null)
                {
                    //skip this one
                }
                else
                {
                    //save the clinic just in case we dont have it in the central db
                    HospitalLocationTO hlto = ato.clinic;
                    long lClinicID = -1;
                    if (hlto != null)
                    {
                        lClinicID = CDataUtils2.ToLong(hlto.id);
                        CClinicDataXfer cd = new CClinicDataXfer(this);
                        status = cd.SaveClinic((long)k_SYSTEM.MDWS, lClinicID, hlto.name);
                    }

                    //facility id same as site id
                    long lSiteID = -1;
                    TaggedText ttFacility = ato.facility;
                    if (ttFacility != null)
                    {
                        lSiteID = CDataUtils2.ToLong(ttFacility.tag);
                    }

                    //load the transfer object
                    CPatientApptItemXfer xfer = new CPatientApptItemXfer();
                    xfer.ApptClinicID = lClinicID;
                    xfer.ApptCurrentStatus = ato.currentStatus;
                    xfer.ApptID = ato.id;
                    xfer.ApptPurpose = ato.purpose;
                    xfer.ApptSiteID = lSiteID;
                    xfer.ApptStatus = ato.status;
                    xfer.ApptText = ato.text;
                    xfer.ApptTimeStamp = ato.timestamp;
                    xfer.ApptTitle = ato.title;
                    xfer.ApptType = ato.type;
                    xfer.ApptVisitID = "";
                    //broke when we switched to mdws instead of devmdws ato.visitId;
                    xfer.DFN = this.DFN;
                    xfer.RegionID = this.RegionID;
                    xfer.SiteID = this.SiteID;

                    //transdfer the appointments
                    CPatientApptXfer apptXfer = new CPatientApptXfer(this);
                    status = apptXfer.SavePatientAppointment((long)k_SYSTEM.MDWS,
                                                             xfer);
                }
            }
        }

        return status;
    }


    /// <summary>
    /// US:6071 transfer patient visits to central database
    /// </summary>
    /// <param name="taas"></param>
    /// <returns></returns>
    public CStatus TransferPatientVisits(TaggedVisitArray taas)
    {
        CStatus status = new CStatus();

        //nothing to do
        if (taas == null || taas.visits == null)
        {
            return status;
        }

        foreach (VisitTO visTO in taas.visits)
        {
            CPatientVisitItemXfer xferItem = new CPatientVisitItemXfer();
            xferItem.DFN = this.DFN;
            xferItem.RegionID = this.RegionID;
            xferItem.SiteID = this.SiteID;

            xferItem.VisitID = visTO.visitId;
            xferItem.VisitLocationID = CDataUtils2.ToLong(visTO.location.id);
            xferItem.VisitLocationName = visTO.location.name;
            xferItem.VisitTimeStamp = visTO.timestamp;
            xferItem.VisitStatus = visTO.status;
            xferItem.VisitType = visTO.type;
            
            CPatientVisitXfer xfer = new CPatientVisitXfer(this);
            xfer.SavePatientVisit((long)k_SYSTEM.MDWS,
                                   xferItem);
        }
        
        return status;
    }

    /// <summary>
    /// US:6006 transfer patient admissions to central database
    /// </summary>
    /// <param name="taas"></param>
    /// <returns></returns>
    public CStatus TransferPatientAdmissions(TaggedInpatientStayArray taas)
    {
        CStatus status = new CStatus();

        //nothing to do
        if (taas == null || taas.stays == null)
        {
            return status;
        }
        
        foreach (InpatientStayTO isTO in taas.stays)
        {
            CPatientAdmissionItemXfer xferItem = new CPatientAdmissionItemXfer();

            //MDWS GetAdmissions gets the selected patients admissions
            xferItem.DFN = this.DFN;
            xferItem.AdmitLocationID = Convert.ToInt32(isTO.location.id);
            xferItem.AdmitLocationName = isTO.location.name;
            xferItem.AdmitType = isTO.type;
            xferItem.AdmitTimeStamp = isTO.admitTimestamp;
            xferItem.AdmitDischargeTimeStamp = isTO.dischargeTimestamp;

            CPatientAdmissionXfer xfer = new CPatientAdmissionXfer(this);
            xfer.SavePatientAdmission((long)k_SYSTEM.MDWS,
                                       xferItem);
        }
        
        return status;
    }
    
    /// <summary>
    /// US:6007 transfer patient consults
    /// </summary>
    /// <param name="taas"></param>
    /// <returns></returns>
    public CStatus TransferPatientConsults(TaggedConsultArrays taas)
    {
        CStatus status = new CStatus();

        //nothing to do
        if (taas == null || taas.arrays == null)
        {
            return status;
        }
        
        foreach (TaggedConsultArray taa in taas.arrays)
        {
            for (int i = 0; i < taa.count; i++)
            {
                ConsultTO ato = taa.consults[i];
                if (ato == null || taas.fault != null)
                {
                    //skip this one
                }
                else
                {
                    //load consult transfer item
                    CPatientConsultItemXfer xferItem = new CPatientConsultItemXfer();
                    xferItem.RegionID = this.RegionID;
                    xferItem.SiteID = this.SiteID;
                    xferItem.DFN = this.DFN;

                    xferItem.ConsultID = ato.id;
                    xferItem.ConsultStatus = ato.status;
                    xferItem.ConsultText = ato.text;
                    xferItem.ConsultTimeStamp = ato.timestamp;
                    xferItem.ConsultTitle = ato.title;
                    
                    //transfer the consult to the central db
                    CPatientConsultXfer xfer = new CPatientConsultXfer(this);
                    status = xfer.SavePatientConsult((long)k_SYSTEM.MDWS,
                                                      xferItem);
                }
            }
        }

        return status;
    }

    /// <summary>
    /// US:838
    /// transfer MDWS wards to the checklist db
    /// </summary>
    /// <param name="txtSpecialties"></param>
    /// <param name="lCount"></param>
    /// <returns></returns>
    public CStatus TransferWards(TaggedHospitalLocationArray HospitalLocArray,
                                 out long lCount)
    {
        //create a status object and check for valid dbconnection
        lCount = 0;
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //check for valid data
        if (HospitalLocArray == null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;

            string strErr = "Error transferring hospital locations, if this error continues please contact your system administrator.";
            status.StatusComment = strErr;
            
            return status;
        }

        //make sure patient is valid (no fault)
        if (HospitalLocArray.fault != null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            string strErr = "Error transferring hospital locations, if this error continues please contact your system administrator.";
            status.StatusComment = strErr;

            //per wasa scan do not return sensitive information to the user.
            //status.StatusComment = HospitalLocArray.fault.message;
            return status;
        }

        //loop over the teams and save each one to the database
        foreach (HospitalLocationTO hlto in HospitalLocArray.locations)
        {
            if (hlto != null)
            {
                lCount++;

                long lWardID =CDataUtils2.ToLong(hlto.id);
                CWardDataXfer wd = new CWardDataXfer(this);
                status = wd.SaveWard((long)k_SYSTEM.MDWS, lWardID, hlto.name);

                if (!status.Status)
                {
                }
            }
        }

        return status;
    }

    /// <summary>
    /// US:5348 transfer user security keys
    /// </summary>
    /// <param name="usk"></param>
    /// <param name="lCount"></param>
    /// <returns></returns>
    public CStatus TransferSecurityKeys( long lUserID,
                                         UserSecurityKeyArray usk,
                                         out long lCount)
    {
        //create a status object and check for valid dbconnection
        lCount = 0;
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //check for valid data
        if (usk == null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            string strErr = "Error transferring security keys, if this error continues please contact your system administrator.";
            status.StatusComment = strErr;


            return status;
        }

        //make sure usk is valid (no fault)
        if (usk.fault != null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;

            string strErr = "Error transferring security keys, if this error continues please contact your system administrator.";
            status.StatusComment = strErr;

            //per wasa scan do not return sensitive information to the user.
            //status.StatusComment = usk.fault.message;
            return status;
        }

        //loop over the teams and save each one to the database
        foreach (UserSecurityKeyTO keyTO in usk.keys)
        {
            if (keyTO != null)
            {
                lCount++;

                long lKeyID =CDataUtils2.ToLong(keyTO.id);
                CSecurityKeyDataXfer keyData = new CSecurityKeyDataXfer(this);
                status = keyData.SaveSecurityKey((long)k_SYSTEM.MDWS, 
                                                 lUserID, 
                                                 lKeyID,
                                                 keyTO.name);

                if (!status.Status)
                {
                }
            }
        }

        return status;
    }

    /// <summary>
    /// US:6048
    /// transfer clinics to the central db
    /// </summary>
    /// <param name="txtSpecialties"></param>
    /// <param name="lCount"></param>
    /// <returns></returns>
    public CStatus TransferClinics(TaggedHospitalLocationArray HospitalLocArray,
                                   string strTarget)
    {
        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //check for valid data
        if (HospitalLocArray == null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            status.StatusComment = "";
            return status;
        }

        //make sure clinic is valid (no fault)
        if (HospitalLocArray.fault != null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            string strErr = "Error transferring clinics, if this error continues please contact your system administrator.";
            status.StatusComment = strErr;

            //per wasa scan do not return sensitive information to the user.
            //status.StatusComment = HospitalLocArray.fault.message;
            return status;
        }

        //build a search string
        string strSearch = strTarget.ToUpper();

        //bool to signal we are done searching
        bool bDone = true;

        //loop over the teams and save each one to the database
        foreach (HospitalLocationTO hlto in HospitalLocArray.locations)
        {
            if (hlto != null)
            {
                 string strCheck = hlto.name.ToUpper();
                 int nPos = strCheck.IndexOf(strSearch);
                 if (nPos != -1)
                 {
                     long lClinicID = CDataUtils2.ToLong(hlto.id);
                     CClinicDataXfer cd = new CClinicDataXfer(this);
                     status = cd.SaveClinic((long)k_SYSTEM.MDWS, lClinicID, hlto.name);
                     bDone = false;
                 }
            }
        }

        if (bDone)
        {
            //match not found so we are done
            status.StatusCode = k_STATUS_CODE.Done;
            status.StatusComment = "";
        }

        return status;
    }

    static public DateTime GetNullDate()
    {
        return new DateTime(0001, 01, 01, 0, 0, 0);
    }

    /// <summary>
    /// get MDWS date without time
    /// </summary>
    /// <param name="strDateTime"></param>
    /// <returns></returns>
    static public DateTime GetMDWSShortDate(string strDateTime)
    {
        if (strDateTime.Length < 8)
        {
            return GetNullDate();
        }

        string strYear = strDateTime.Substring(0, 4);
        string strMonth = strDateTime.Substring(4, 2);
        string strDay = strDateTime.Substring(6, 2);

        return new System.DateTime(Convert.ToInt32(strYear),
                                    Convert.ToInt32(strMonth),
                                    Convert.ToInt32(strDay),
                                    0,
                                    0,
                                    0);
    }

    /// <summary>
    /// US:838
    /// Transfers 1 patient and adds the patient to a team, ward etc... as needed
    /// </summary>
    /// <param name="tpa"></param>
    /// <param name="lCount"></param>
    /// <returns></returns>
    public CStatus TransferPatient(string strDUZ,
                                   long lTeamID,
                                   long lSpecialtyID,
                                   long lWardID,
                                   long lClinicID,
                                   PatientTO toPatient)
    {
        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //check for valid patient
        if (toPatient == null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            status.StatusComment = "Patient is null";

            return status;
        }

        //make sure patient is valid (no fault)
        if (toPatient.fault != null)
        {
            status.Status = false;
            status.StatusCode = k_STATUS_CODE.Failed;
            status.StatusComment = "Patient has a fault";

            return status;
        }

        if (String.IsNullOrEmpty(toPatient.ssn) ||
            toPatient.ssn.IndexOf("*") != -1)
        {
            PatientTO toPatSel = GetMDWSSOAPClient().select(toPatient.localPid);
            toPatient = toPatSel;
        }

        //our patient item
        CPatientDataItemXfer pdi = new CPatientDataItemXfer();
     
        //formatted 19450309
        pdi.DOB = GetMDWSShortDate(toPatient.dob);
         
        //name
        pdi.FullName = toPatient.patientName;
        string[] splitFullName = toPatient.patientName.Split(',');
        if (splitFullName[0] != null)
        {
            pdi.LastName = splitFullName[0];
        }

        if (splitFullName.Length > 1)
        {
            if (splitFullName[1] != null)
            {
                pdi.FirstName = splitFullName[1];
            }
        }

        pdi.MI = String.Empty;

        //gender
        //default gender to unknown
        pdi.Sex = k_SEX.UNKNOWN;
        if (toPatient.gender == null)
        {
            pdi.Sex = k_SEX.UNKNOWN;
        }
        else
        {
            if (toPatient.gender.ToLower().IndexOf("unk") != -1)
            {
                pdi.Sex = k_SEX.UNKNOWN;
            }
            else
            {
                if (toPatient.gender.ToUpper() == "M")
                {
                    pdi.Sex = k_SEX.MALE;
                }
                if (toPatient.gender.ToUpper() == "F")
                {
                    pdi.Sex = k_SEX.FEMALE;
                }
            }
        }

        pdi.SSN = toPatient.ssn;
        pdi.PatientID = toPatient.localPid;

        //save the patient to our db
        CPatientDataXfer dta = new CPatientDataXfer(this);
        status = dta.SavePatient(CDataUtils2.ToLong(strDUZ), 
                                 lTeamID, 
                                 lSpecialtyID, 
                                 lWardID, 
                                 lClinicID,
                                 (long)k_SYSTEM.MDWS, 
                                 pdi);
        
        return status;
    }
    

    /// <summary>
    /// US:838
    /// Transfers a patient array to our db
    /// </summary>
    /// <param name="tpa"></param>
    /// <param name="lCount"></param>
    /// <returns></returns>
    public CStatus TransferPatientArray(string strDUZ,
                                        long lTeamID,
                                        long lSpecialtyID,
                                        long lWardID,
                                        long lClinicID,
                                        TaggedPatientArray tpa,
                                        out long lCount)
    {
        //initialize parameters
        lCount = 0;

        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //nothing to do
        if (tpa == null || tpa.patients == null)
        {
            return status;
        }

        //check to see if there is anything to transfer
        if (tpa.count < 1)
        {
            return status;
        }
        
        foreach (PatientTO pat in tpa.patients)
        {
            //additional demographics
            DemographicSetTO[] demog = pat.demographics;
            
            CStatus statXfer = new CStatus();
            statXfer = TransferPatient(strDUZ, 
                                       lTeamID, 
                                       lSpecialtyID, 
                                       lWardID,
                                       lClinicID,
                                       pat);
            lCount++;
            if (!statXfer.Status)
            {
                status.Status = false;
                status.StatusCode = k_STATUS_CODE.Failed;
                status.StatusComment = "One or more of your patient(s) could not be transfered!";
                //note: dont return yet, try the rest of the patients
            }
        }
        
        return status;
    }

    /// <summary>
    /// US:5348 US:6047 transfer a MDWS user array to our db
    /// </summary>
    /// <param name="ua"></param>
    /// <param name="lCount"></param>
    /// <returns></returns>
    public CStatus TransferUserArray( UserArray ua, string strSearch)
    {
        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //nothing to do
        if (ua == null || ua.users == null)
        {
            return status;
        }


        //check to see if there is anything to transfer
        if (ua.count < 1)
        {
            return status;
        }

        //make the search upper for compare below, MDWS seems to want the text ucase
        string strSrch = strSearch.ToUpper();

        //bool to signal we are done searching
        bool bDone = true;

        //loop and transferr
        foreach (UserTO toUser in ua.users)
        {           
            long lUserID = 0;

            //only transfer records that actually match our search criteria
            string strCompare = toUser.name.ToUpper();
            int nPos = strCompare.IndexOf(strSrch);
            if (nPos != -1)
            {
                CStatus statXfer = new CStatus();
                statXfer = TransferUser(toUser,
                                        this.RegionID,
                                        this.SiteID,
                                        out lUserID);
                bDone = false;
            }
        }

        //if no match found in entire group we are done
        if(bDone)
        {
            status.StatusCode = k_STATUS_CODE.Done;
            status.StatusComment = "";
        }

        return status;
    }

    /// <summary>
    /// US:5348 US:6047
    /// transfer MDWS user data to the database
    /// </summary>
    /// <param name="toUser"></param>
    /// <param name="lUserID"></param>
    /// <returns></returns>
    public CStatus TransferUser(UserTO toUser, long lRegionID, long lSiteID, out long lFXUserID)
    {
        //initialize parameters
        lFXUserID = 0;

        //create a status object and check for valid dbconnection
        CStatus status = DBConnValid();
        if (!status.Status)
        {
            return status;
        }

        //nothing to do
        if (toUser == null)
        {
            return status;
        }

        string strTitle = String.Empty;
        if (!String.IsNullOrEmpty(toUser.title))
        {
            strTitle = toUser.title;
        }        

        string strUserClass = String.Empty;
        if (!String.IsNullOrEmpty(toUser.userClass))
        {
            strUserClass = toUser.userClass;
        }

        string strGreeting = "";
        if (!String.IsNullOrEmpty(toUser.greeting))
        {
            strGreeting = toUser.greeting;
        }

        //name is formatted: VEHU,ONE 
        string strUserName = "";
        string strFirstName = "";
        string strLastName = "";
        if (!String.IsNullOrEmpty(toUser.name))
        {
            strUserName = toUser.name;
         
            //splie the name into last and first
            string[] splitName = strUserName.Split(',');
            if (splitName.Length > 0)
            {
                strLastName = splitName[0];
            }
            if (splitName.Length > 1)
            {
                strFirstName = splitName[1];
            }
        }

        string strFirstLastName = strFirstName + " " + strLastName;

        string strDUZ = "";
        if (!String.IsNullOrEmpty(toUser.DUZ))
        {
            strDUZ = toUser.DUZ;
        }
       
        //transfer this user to the central db.
        CUserDataXfer UserXfer = new CUserDataXfer(this);
        status = UserXfer.SaveUser((long)k_SYSTEM.MDWS,
                                  strDUZ,
                                  strLastName,
                                  strFirstName,
                                  strFirstLastName,
                                  strGreeting,
                                  strTitle,
                                  lRegionID,
                                  lSiteID,
                                  strUserClass,
                                  out lFXUserID);
        
        
        return status;
    }
}
